<?php

namespace App\Http\Controllers;

use App\Http\Requests\MailStoreRequest;
use App\Models\AddressBook;
use App\Models\CountryCode;
use App\Models\Mailbox;
use App\Models\MailboxAttachment;
use App\Models\SubscriptionServicePackage;
use App\Models\User;
use App\Services\AddressBookService;
use App\Services\MailBoxService;
use App\Services\UserService;
use Faker\Provider\ar_EG\Address;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Mail;
use PHPUnit\Framework\Constraint\Count;

class MailboxController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function mailbox(Request $request)
    {
        $mail = Mailbox::with('addressBook')
            ->select('mailboxes.*', 'mailbox_attachments.attachment')
            ->leftJoin('mailbox_attachments', 'mailboxes.id', '=', 'mailbox_attachments.mailbox_id')
            ->where('client_id', auth()->user()->id)
            ->orderBy('mailboxes.id', 'desc');

        if ($request->has('search')) {
            $mail->where('title', 'like', '%' . $request->search . '%')
                ->orWhere('short_desc', 'like', '%' . $request->search . '%');
        }

        $mail = $mail->paginate(10);


        return view('app.mailbox.index', compact('mail'));
    }
    public function mailList(Request $request)
    {
        $mails = Mailbox::with('client')
            ->select('mailboxes.*', 'address_books.name as address_book_name', 'mailbox_attachments.attachment')
            ->leftJoin('address_books', 'mailboxes.mail_from', '=', 'address_books.id')
            ->leftJoin('mailbox_attachments', 'mailboxes.id', '=', 'mailbox_attachments.mailbox_id')
            ->orderBy('mailboxes.id', 'desc');

        if ($request->type == 'pending_attachment') {
            $mails->where('scan_permission', 1)
                ->whereNull('is_scanned');
        }

        $mails = $mails->get();



        return view('app.mailbox.mail-list', compact('mails'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function mailEntry()
    {

        return view('app.mailbox.mail-entry');
    }
    public function mailEntryForm(Request $request, UserService $userService)
    {

        $user = User::where('user_id', $request->user_id)->where('user_type_id', 1)->first();
        if (!$user) {
            return response()->json(['message' => 'User not found'], 404);
        }

        $address_book = AddressBook::where('client_id', $user->id)->get();
        $country = CountryCode::all();

        $user = $userService->getUserProfile($user->id);

        $html = view('app.mailbox.mail-entry-form', compact('user', 'address_book', 'country', 'user'))->render();
        return response()->json(['html' => $html], 200);
    }

    public function mailEntryFormStore(MailStoreRequest $request, MailBoxService $mailBoxService)
    {

        if ($request->scan_permission == 1) {
            $is_scanned = now();
        } else {
            $is_scanned = null;
        }

        try {

            DB::beginTransaction();
            if ($request->address_id == 0) {
                $address_book_service = new AddressBookService();
                $address_book_id = $address_book_service->store($request, $request->user_id)->id;
            } else {
                $address_book_id = $request->address_id;
            }

            $address = AddressBook::leftJoin('country_codes as cpc', 'address_books.country_id', '=', 'cpc.id')
                ->leftJoin('country_codes as cc', 'address_books.country_code_id', '=', 'cc.id')
                ->select('address_books.*', 'cpc.country_code as country_phone_code', 'cc.country_name as country_name')
                ->where('address_books.id', $address_book_id)
                ->first();

            $ssp = SubscriptionServicePackage::where('client_id', $request->user_id)->where('status', 'current')->first()->id;

            $mail_id =  $mailBoxService->store($request, $address, $address_book_id, $is_scanned, $ssp);

            if ($request->has('attachment')) {
                $attachment = fileUpload($request->attachment, 'mailbox/' . now()->format('Y'));
                $attachment_title = $request->user_id . '_' . $mail_id . '.' . $request->attachment->getClientOriginalExtension();
                $mailBoxService->storeAttachment('mailbox/' . now()->format('Y') . '/' . $attachment, $attachment_title);
            }

            DB::commit();
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', $e->getMessage());
        }

        return back()->with('success', 'Mail has been successfully entered')->with('mail_id', $mail_id);
    }

    public function mailDetail($id, MailBoxService $mailBoxService)
    {

        $mail_details = $mailBoxService->getMailDetails($id);

        $html = view('app.mailbox.mail-detail')->with('mail_details', $mail_details)->render();

        return response()->json(['html' => $html], 200);
    }

    public function mailDetailClient(Mailbox $mailbox, MailBoxService $mailBoxService)
    {

        if (!$mailbox->is_opened) {
            $mailbox->update(['is_opened' => now()]);
        }

        $mail_details = $mailBoxService->getMailDetails($mailbox->id);


        return view('app.mailbox.mail-detail-client', compact('mail_details'));
    }

    public function mailScanPermissionUpdate(Request $request)
    {
        $mail = Mailbox::where('id', $request->id)->first();
        $mail->update(['scan_permission' => 1, 'scan_permission_at' => now()]);

        return response()->json(['message' => 'Scan permission updated successfully'], 200);
    }
}
